%% Script that analyses NAL-DCT scores
clear,clc,close all

%% Process data in a table
S = readtable('NALDCT_Scores.xlsx');        % Raw Scores
ID = sort(repmat((1:32)',6,1),'ascend');    % [1-16] UHL  |  [17-32] NH
Run = repmat((1:6)',32,1);
Score = nan(32*6,1);
for i=1:32              % Loop in Participants
    for j=1:6           % Loop in Tests
        Score((i-1)*6+j) = sum(table2array(S((j-1)*10+1:j*10,i+2)))*10;
    end
end
Group = [repmat('UHL',96,1) ; repmat('NH_',96,1)];
T = table(nominal(ID),Run,Score,nominal(Group),'VariableNames',...
    {'ID','Run','Score','Group'});

%% Check for normality
% Get average %-correct per participant
AvScore = nan(16,2);
for i=1:16
    AvScore(i,1) = mean(T.Score((i-1)*6+1:i*6));            % Column 1: UHL
    AvScore(i,2) = mean(T.Score((i+15)*6+1:(i+16)*6));      % Column 2: NH
end
save('NALDCT_AvScores','AvScore')

% p-values > 0.05   =>   Data is considered normally distributed
[~,p_UHL] = lillietest(AvScore(:,1));
[~,p_NH]  = lillietest(AvScore(:,2));

%% Linear mixed-effects model
LME = fitlme(T,'Score ~ Group + (1|ID)','FitMethod','REML');
disp(LME)
anova(LME)

%% Figure
figure('PaperSize',[9 7]),grid on,hold on,box on
cmap = colormap('lines');
FS = 10;    disp = 0.1;     MS = 5;     LW = 1;
plot(1+disp*randn(16,1),AvScore(:,2),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(1,:),'MarkerEdgeColor','w')
plot(2+disp*randn(16,1),AvScore(:,1),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(2,:),'MarkerEdgeColor','w')
plot([1.0,2.0],[95,95],'LineWidth',LW,'Color','k')
plot([1.0,1.0],[93,95],'LineWidth',LW,'Color','k')
plot([2.0,2.0],[93,95],'LineWidth',LW,'Color','k')
text(1.5,96.5,'*','FontSize',FS+10,...
    'HorizontalAlignment','center','color','k')
text(1,52,'73.7 %','FontSize',FS-1,'HorizontalAlignment','center')
text(2,50,'68.2 %','FontSize',FS-1,'HorizontalAlignment','center')
text(1,5.8,'***','FontSize',FS+10,'HorizontalAlignment','center')
axis([0.5 2.5 40 104])
set(gca,'xtick',1:2,'xticklabel',{'NH','UHL'},'FontSize',FS)
set(gca,'ytick',40:10:100,'yticklabel',{'40%','50%','60%','70%','80%',...
    '90%','100%'},'FontSize',FS)
ylabel('NAL-DCT score','FontSize',FS)
title('Speech-in-noise comprehension','FontSize',FS)

orient tall
print('-dpng','-r600','Figure4')
print('-dpdf','-r600','Figure4')
close all
